package ga.core.individual;

import ga.core.validation.GAContext;
import ga.core.validation.IValidator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

/**
 * {@link ArrayList} for individuals. Adds sorting and validation functions.
 * 
 * @param <T>
 *          The generic type of individuals.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
@SuppressWarnings("serial")
public class IndividualList<T extends IIndividual<T>> extends ArrayList<T> {

  /**
   * Creates a new list.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public IndividualList() {

  }

  /**
   * Creates a new list by copying the given collection.
   * 
   * @param c
   *          Collection to copy.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public IndividualList(final Collection<? extends T> c) {
    super(c);
  }

  /**
   * Sorts the list by using {@link IndividualComparator}.
   * 
   * @param descending
   *          Sort descending if <code>true</code>, ascending otherwise.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void sort(final boolean descending) {
    boolean ok = true;
    do {
      try {
        Collections.sort(this, new IndividualComparator<T>(descending));
      } catch (final java.lang.IllegalArgumentException e) {
        ok = false;
      }
    } while (!ok);
  }

  /**
   * Validates the list with a given validator.
   * 
   * @param validator
   *          The validator to use, nothing will happen if it is
   *          <code>null</code>.
   * @param context
   *          The GA context.
   * @return <code>true</code> if all the individuals in the list are valid.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public boolean isValid(final IValidator<T> validator, final GAContext context) {
    if (validator != null) {
      for (final T individual : this) {
        if (!validator.isValid(individual, context)) {
          return false;
        }
      }
    }

    return true;
  }
}
